/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A zoom selector.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgzoomselector.h"
#include "skgservices.h"

SKGZoomSelector::SKGZoomSelector(QWidget* iParent)
    : QWidget(iParent), m_resetValue(0)
{
    ui.setupUi(this);

    // Set icons
    ui.kZoomIn->setIcon(SKGServices::fromTheme(QStringLiteral("zoom-in")));
    ui.kZoomOut->setIcon(SKGServices::fromTheme(QStringLiteral("zoom-out")));
    ui.kZoomOriginal->setIcon(SKGServices::fromTheme(QStringLiteral("zoom-original")));

    connect(ui.kZoomOriginal, &QToolButton::clicked, this, &SKGZoomSelector::initializeZoom);
    connect(ui.kZoomOut, &QToolButton::clicked, this, &SKGZoomSelector::zoomOut);
    connect(ui.kZoomIn, &QToolButton::clicked, this, &SKGZoomSelector::zoomIn);
    connect(ui.kZoomSlider, &QSlider::valueChanged, this, &SKGZoomSelector::onZoomChangedDelayed);

    // Init timer
    m_timer.setSingleShot(true);
    connect(&m_timer, &QTimer::timeout, this, &SKGZoomSelector::onZoomChanged, Qt::QueuedConnection);
}

SKGZoomSelector::~SKGZoomSelector()
    = default;

void SKGZoomSelector::setValue(int iValue, bool iEmitEvent)
{
    bool previous = ui.kZoomSlider->blockSignals(!iEmitEvent);
    ui.kZoomSlider->setValue(iValue);
    if (ui.kZoomOriginal != nullptr) {
        ui.kZoomOriginal->setChecked(iValue == m_resetValue);
    }
    ui.kZoomSlider->blockSignals(previous);
}

int SKGZoomSelector::value() const
{
    return ui.kZoomSlider->value();
}

int SKGZoomSelector::resetValue() const
{
    return m_resetValue;
}

void SKGZoomSelector::setResetValue(int iValue)
{
    m_resetValue = iValue;
}

void SKGZoomSelector::zoomIn()
{
    ui.kZoomSlider->setValue(ui.kZoomSlider->value() + 1);
}

void SKGZoomSelector::zoomOut()
{
    ui.kZoomSlider->setValue(ui.kZoomSlider->value() - 1);
}

void SKGZoomSelector::initializeZoom()
{
    ui.kZoomSlider->setValue(m_resetValue);
}

void SKGZoomSelector::onZoomChangedDelayed()
{
    m_timer.start(300);
}

void SKGZoomSelector::onZoomChanged()
{
    int val = ui.kZoomSlider->value();
    if (ui.kZoomIn != nullptr) {
        ui.kZoomIn->setEnabled(val < 10);
    }
    if (ui.kZoomOut != nullptr) {
        ui.kZoomOut->setEnabled(val > -10);
    }
    if (ui.kZoomOriginal != nullptr) {
        ui.kZoomOriginal->setChecked(val == m_resetValue);
    }

    Q_EMIT changed(val);
}


