
/* 
A* -------------------------------------------------------------------
B* This file contains source code for the PyMOL computer program
C* copyright 1998-2000 by Warren Lyford Delano of DeLano Scientific. 
D* -------------------------------------------------------------------
E* It is unlawful to modify or remove this copyright notice.
F* -------------------------------------------------------------------
G* Please see the accompanying LICENSE file for further information. 
H* -------------------------------------------------------------------
I* Additional authors of this source file include:
-* 
-* 
-*
Z* -------------------------------------------------------------------
*/
#include"os_python.h"

#include"os_predef.h"
#include"os_std.h"
#include"os_gl.h"

#include"OOMac.h"
#include"ObjectCGO.h"
#include"Base.h"
#include"MemoryDebug.h"
#include"CGO.h"
#include"Scene.h"
#include"Setting.h"
#include"PConv.h"
#include"main.h"
#include"Color.h"
#include"VFont.h"
#include"ShaderMgr.h"

static PyObject *ObjectCGOStateAsPyList(ObjectCGOState * I)
{
  PyObject *result = NULL;

  result = PyList_New(1);
  if(I->origCGO)
    PyList_SetItem(result, 0, CGOAsPyList(I->origCGO));
  else
    PyList_SetItem(result, 0, PConvAutoNone(NULL));
  return (PConvAutoNone(result));

}

static PyObject *ObjectCGOAllStatesAsPyList(ObjectCGO * I)
{
  PyObject *result = NULL;
  int a;
  result = PyList_New(I->NState);
  for(a = 0; a < I->NState; a++) {
    PyList_SetItem(result, a, ObjectCGOStateAsPyList(I->State + a));
  }
  return (PConvAutoNone(result));
}

static int ObjectCGOStateFromPyList(PyMOLGlobals * G, ObjectCGOState * I, PyObject * list,
                                    int version)
{
  int ok = true;
  int ll, pl = 0;
  PyObject *tmp;
  if(ok)
    ok = (list != NULL);
  if(ok)
    ok = PyList_Check(list);
  if(ok)
    ll = PyList_Size(list);
  CGOFree(I->origCGO);
  /* TO SUPPORT BACKWARDS COMPATIBILITY...
     Always check ll when adding new PyList_GetItem's */
  if(ok && ll==2) {
    tmp = PyList_GetItem(list, 0);
    if(tmp == Py_None)
      I->origCGO = NULL;
    else {
      ok = ((I->origCGO = CGONewFromPyList(G, tmp, version, 1)) != NULL);
    }
    pl++;
  }
  if(ok && !I->origCGO) {
    tmp = PyList_GetItem(list, pl);
    if(tmp == Py_None)
      I->origCGO = NULL;
    else {
      ok = ((I->origCGO = CGONewFromPyList(G, tmp, version, 0)) != NULL);
    }
  }
  return (ok);
}

static int ObjectCGOAllStatesFromPyList(ObjectCGO * I, PyObject * list, int version)
{
  int ok = true;
  int a;
  VLACheck(I->State, ObjectCGOState, I->NState);
  if(ok)
    ok = PyList_Check(list);
  if(ok) {
    for(a = 0; a < I->NState; a++) {
      ok =
        ObjectCGOStateFromPyList(I->Obj.G, I->State + a, PyList_GetItem(list, a),
                                 version);
      if(!ok)
        break;
    }
  }
  return (ok);

}

int ObjectCGONewFromPyList(PyMOLGlobals * G, PyObject * list, ObjectCGO ** result,
                           int version)
{
  int ok = true;
  ObjectCGO *I = NULL;
  (*result) = NULL;
  if(ok)
    ok = (list != NULL);
  if(ok)
    ok = PyList_Check(list);

  I = ObjectCGONew(G);
  if(ok)
    ok = (I != NULL);

  if(ok)
    ok = ObjectFromPyList(G, PyList_GetItem(list, 0), &I->Obj);
  if(ok)
    ok = PConvPyIntToInt(PyList_GetItem(list, 1), &I->NState);
  if(ok)
    ok = ObjectCGOAllStatesFromPyList(I, PyList_GetItem(list, 2), version);
  if(ok) {
    (*result) = I;
    ObjectCGORecomputeExtent(I);
  } else {
    /* cleanup? */
  }
  return (ok);
}

PyObject *ObjectCGOAsPyList(ObjectCGO * I)
{
  PyObject *result = NULL;

  result = PyList_New(3);
  PyList_SetItem(result, 0, ObjectAsPyList(&I->Obj));
  PyList_SetItem(result, 1, PyInt_FromLong(I->NState));
  PyList_SetItem(result, 2, ObjectCGOAllStatesAsPyList(I));

  return (PConvAutoNone(result));
}


/*========================================================================*/

void ObjectCGOFree(ObjectCGO * I)
{
  int a;
  for(a = 0; a < I->NState; a++) {
    CGOFree(I->State[a].renderCGO);
    CGOFree(I->State[a].origCGO);
  }
  VLAFreeP(I->State);
  ObjectPurge(&I->Obj);
  OOFreeP(I);
}


/*========================================================================*/

void ObjectCGORecomputeExtent(ObjectCGO * I)
{
  float mx[3], mn[3];
  int extent_flag = false;
  int a;
  int has_normals = 0;
  CGO *cgo;
  for(a = 0; a < I->NState; a++){
    cgo = I->State[a].origCGO;
    if (!cgo){
      cgo = I->State[a].renderCGO;
    }
    if(cgo) {
      if(CGOGetExtent(cgo, mn, mx)) {
        if(!extent_flag) {
          extent_flag = true;
          copy3f(mx, I->Obj.ExtentMax);
          copy3f(mn, I->Obj.ExtentMin);
        } else {
          max3f(mx, I->Obj.ExtentMax, I->Obj.ExtentMax);
          min3f(mn, I->Obj.ExtentMin, I->Obj.ExtentMin);
        }
      }
      if (!has_normals && cgo && CGOHasNormals(cgo)){
	has_normals = 1;
      }
    }
  }
  I->Obj.ExtentFlag = extent_flag;
  SettingCheckHandle(I->Obj.G, &I->Obj.Setting);
  SettingSet_i(I->Obj.Setting, cSetting_cgo_lighting, has_normals);
}


/*========================================================================*/
static void ObjectCGOInvalidate(ObjectCGO * I, int rep, int level, int state)
{
  ObjectCGOState *sobj = NULL;

  if(state < 0) {
    int a;
    for(a = 0; a < I->NState; a++) {
      sobj = I->State + a;
      if (sobj->renderCGO){
	CGOFree(sobj->renderCGO);	      
	sobj->renderCGO = 0;
      }
    }
  } else {
    if((state >= 0) && (state < I->NState)) {
      sobj = I->State + state;
      if (sobj->renderCGO){
	CGOFree(sobj->renderCGO);	      
	sobj->renderCGO = 0;
      }
    }
  }
}


/*========================================================================*/

static void ObjectCGOUpdate(ObjectCGO * I)
{
  int a;
  for(a = 0; a < I->NState; a++) {
    ObjectCGOState *ocs = I->State + a;
    if (ocs->renderCGO){
      CGOFree(ocs->renderCGO);	      
      ocs->renderCGO = 0;
    }
  }
  SceneInvalidate(I->Obj.G);    /* needed ? */
}


/*========================================================================*/

static int ObjectCGOGetNState(ObjectCGO * I)
{
  return (I->NState);
}

static void ObjectCGORenderState(PyMOLGlobals * G, int pass, CRay *ray, Picking **pick, ObjectCGO * I, RenderInfo * info, ObjectCGOState *sobj, const float *color, ObjectGadgetRamp *ramp, int use_shader, bool cgo_lighting){
  if(ray) {
    if(sobj) {
      if(sobj->origCGO){
        CGO *cgo = sobj->origCGO, *cgo_copy = NULL;
        if (cgo_lighting && CGOHasAnyTriangleVerticesWithoutNormals(cgo)) {
          cgo = cgo_copy = CGOGenerateNormalsForTriangles(cgo);
        }
        CGORenderRay(cgo, ray, info, color, ramp, I->Obj.Setting, NULL);
        CGOFree(cgo_copy);
      }
    }
  } else if(G->HaveGUI && G->ValidContext && pass) {
    if(pick) { // no picking yet
    } else {
      bool pass_is_opaque = (pass > 0);
      if(sobj && ((sobj->hasTransparency ^ pass_is_opaque) || (sobj->hasOpaque == pass_is_opaque))){
	{
	  CShaderPrg *shaderPrg;
	  int two_sided_lighting = SettingGet_i(G, I->Obj.Setting, NULL, cSetting_two_sided_lighting);
          bool backface_cull = SettingGet_i(G, I->Obj.Setting, NULL, cSetting_backface_cull);
	  if (two_sided_lighting<0){
	    two_sided_lighting = !cgo_lighting;
	  }
	  two_sided_lighting &= cgo_lighting;  // only set two_sided_lighting if cgo_lighting is set
#ifndef PURE_OPENGL_ES_2
	  if (cgo_lighting){
	    glEnable(GL_LIGHTING);
	  } else {
	    glDisable(GL_LIGHTING);
	  }
	  if (two_sided_lighting){
	    if (use_shader)
	      glEnable(GL_VERTEX_PROGRAM_TWO_SIDE);
	    GLLIGHTMODELI(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
	  } else {
	    if (use_shader)
	      glDisable(GL_VERTEX_PROGRAM_TWO_SIDE);
	    GLLIGHTMODELI(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
	  }
#endif
	  if (backface_cull){
	    glCullFace(GL_BACK);
	    glEnable(GL_CULL_FACE);
	  }

	  if (use_shader){
	    shaderPrg = G->ShaderMgr->Enable_DefaultShader(info->pass);
	    if (!shaderPrg) return;
	    shaderPrg->SetLightingEnabled(cgo_lighting);
	    shaderPrg->Set1i("two_sided_lighting_enabled", two_sided_lighting);
	    sobj->renderCGO->use_shader = use_shader;
	    sobj->renderCGO->debug = SettingGetGlobal_i(G, cSetting_cgo_debug);
	    CGORenderGL(sobj->renderCGO, color, I->Obj.Setting, NULL, info, NULL);
	    shaderPrg->Disable();
	  } else {
	    sobj->renderCGO->use_shader = use_shader;
	    sobj->renderCGO->debug = SettingGetGlobal_i(G, cSetting_cgo_debug);
	    CGORenderGL(sobj->renderCGO, color, I->Obj.Setting, NULL, info, NULL);
	  }

	    if (backface_cull){
	      glDisable(GL_CULL_FACE);
	    }
#ifndef PURE_OPENGL_ES_2
	    if (two_sided_lighting){
	      if (use_shader)
		glDisable(GL_VERTEX_PROGRAM_TWO_SIDE);
	      GLLIGHTMODELI(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
	    }
	    if (!cgo_lighting){
	      glEnable(GL_LIGHTING);
	    }
#endif
	}
      }
    }
  }
}

static void ObjectCGOGenerateCGO(PyMOLGlobals * G, ObjectCGO * I, ObjectCGOState *sobj, bool use_shader, bool cgo_lighting, const float *color, ObjectGadgetRamp *ramp, int state)
{
  if (sobj->renderCGO && 
      ((use_shader ^ sobj->renderWithShaders) ||
       (cgo_lighting ^ sobj->cgo_lighting))){
    // if renderWithShaders doesn't match use_shader, clear CGO and re-generate
    CGOFree(sobj->renderCGO);
    sobj->renderCGO = 0;
  }
  if (!sobj->renderCGO){
    CGO *convertcgo = NULL;
    float colorWithA[4];
    short someLinesWithoutNormals = 0;
    CGO *preOpt = NULL;
    if (color){
      colorWithA[0] = color[0]; colorWithA[1] = color[1]; colorWithA[2] = color[2];
    } else {
      colorWithA[0] = 1.f; colorWithA[1] = 1.f; colorWithA[2] = 1.f;
    }
    colorWithA[3] = 1.f - SettingGet_f(G, I->Obj.Setting, NULL, cSetting_cgo_transparency);
    preOpt = sobj->origCGO;

    bool hasTransparency = (colorWithA[3] < 1.f || CGOHasTransparency(preOpt));
    bool hasOpaque = (colorWithA[3] == 1.f || CGOHasOpaque(preOpt));

    CGO *allCylinders = NULL;
    CGO *allSpheres = NULL;
    {
      CGO *convertcgo = NULL;
      bool freePreOpt = false;
      if (cgo_lighting){
	if (CGOHasAnyTriangleVerticesWithoutNormals(preOpt)){
	  // we only need normals if cgo_lighting is on
	  preOpt = CGOGenerateNormalsForTriangles(preOpt);
	  freePreOpt = true;
	}
	someLinesWithoutNormals = CGOHasAnyLineVerticesWithoutNormals(preOpt);
	if (!use_shader && someLinesWithoutNormals){
	  // if some lines without normals, turn lighting off on lines
	  CGO *preOpt2 = preOpt;
          preOpt->use_shader = use_shader;
	  preOpt = CGOTurnLightingOnLinesOff(preOpt);
          CGOStop(preOpt);
	  if (freePreOpt){
	    CGOFree(preOpt2);
	  }
	  freePreOpt = true;
	}
      }
      convertcgo = CGONew(G);
      CGOColorv(convertcgo, colorWithA);
      CGOAlpha(convertcgo, colorWithA[3]);
      CGOAppend(convertcgo, preOpt);
      if (freePreOpt){
        CGOFree(preOpt);
      }
      if (use_shader){
        bool t_mode_3 = SettingGetGlobal_i(G, cSetting_transparency_mode)==3;
        if ((t_mode_3 || !hasTransparency)
            && G->ShaderMgr->Get_DefaultSphereShader(0)
            && G->ShaderMgr->Get_CylinderShader(0))
        {
          if (CGOHasCylinderOperations(convertcgo)){
            CGO *newCGO = NULL;
            allCylinders = CGONew(G);
            CGOEnable(allCylinders, GL_CYLINDER_SHADER);
            newCGO = CGOConvertShaderCylindersToCylinderShader(convertcgo, allCylinders);
            CGOAppendNoStop(allCylinders, newCGO);
            CGOFreeWithoutVBOs(newCGO);
            CGODisable(allCylinders, GL_CYLINDER_SHADER);
            CGOStop(allCylinders);
            CGO *allButCylinders = CGONew(G);
            CGOFilterOutCylinderOperationsInto(convertcgo, allButCylinders);
            CGOStop(allButCylinders);
            CGOFree(convertcgo);
            convertcgo = allButCylinders;
          }
          if (CGOHasOperationsOfType(convertcgo, CGO_SPHERE)){
            CGO *allButSpheres = CGONew(G);
            allSpheres = CGOOptimizeSpheresToVBONonIndexed(convertcgo, 0, true, allButSpheres);
            if (allSpheres){
              CGOFree(convertcgo);
              CGOStop(allButSpheres);
              convertcgo = allButSpheres;
            } else {
              CGOFree(allButSpheres);
            }
          }
          preOpt = CGOSimplify(convertcgo, 0);
        } else {
          preOpt = CGOSimplifyNoCompress(convertcgo, 0) ;
        }
      } else {
	preOpt = CGOSimplifyNoCompress(convertcgo, 0) ;
      }
      CGOFree(convertcgo);
    }

    if (ramp){
      convertcgo = CGOColorByRamp(G, preOpt, ramp, state, I->Obj.Setting);
      CGOFree(preOpt);
      preOpt = convertcgo;
    }
    if (use_shader){
      if(preOpt && preOpt->has_begin_end){
	CGO *convertcgo = CGOCombineBeginEnd(preOpt, 0);
	CGOFree(preOpt);
	preOpt = convertcgo;
      }
      sobj->hasTransparency = hasTransparency;
      sobj->hasOpaque = hasOpaque;
      convertcgo = CGOOptimizeToVBOIndexedWithColorEmbedTransparentInfo(preOpt, 0, colorWithA, false);
      if (someLinesWithoutNormals){
        // if some lines without normals, turn lighting off on lines
        convertcgo->use_shader = use_shader;
        CGO *convertcgo2 = CGOTurnLightingOnLinesOff(convertcgo);
        CGOStop(convertcgo2);
        CGOFreeWithoutVBOs(convertcgo);
        convertcgo = convertcgo2;
      }
      if (allCylinders){
        CGOAppendNoStop(convertcgo, allCylinders);
        CGOFreeWithoutVBOs(allCylinders);
      }
      if (allSpheres){
        CGOAppendNoStop(convertcgo, allSpheres);
        CGOFreeWithoutVBOs(allSpheres);
      }
      CGOStop(convertcgo);
      sobj->renderCGO = convertcgo;
    } else {
      unsigned char hasTrans = CGOHasTransparency(preOpt);
      CGOFree(convertcgo);
      if (hasTrans){
	CGO *convertcgo ;
	convertcgo = CGOConvertTrianglesToAlpha(preOpt);
	sobj->renderCGO = convertcgo;
	sobj->renderCGO->render_alpha = 2;
      } else {
	sobj->renderCGO = CGOSimplify(preOpt, 0); // copy, for now
      }
      sobj->hasTransparency = hasTrans;
      sobj->hasOpaque = CGOHasOpaque(preOpt);
    }
    CGOFree(preOpt);
    sobj->renderWithShaders = use_shader;
    sobj->cgo_lighting = cgo_lighting;
  }
}
/*========================================================================*/

static void ObjectCGORender(ObjectCGO * I, RenderInfo * info)
{
  PyMOLGlobals *G = I->Obj.G;
  int state = info->state;
  CRay *ray = info->ray;
  Picking **pick = info->pick;
  int pass = info->pass;
  ObjectCGOState *sobj = NULL;
  const float *color = NULL;
  bool use_shader = false, cgo_lighting = false;
  ObjectGadgetRamp *ramp = NULL;
  
  use_shader = SettingGetGlobal_b(G, cSetting_cgo_use_shader) &
    SettingGetGlobal_b(G, cSetting_use_shaders);
  cgo_lighting = SettingGet_i(G, I->Obj.Setting, NULL, cSetting_cgo_lighting);

  ObjectPrepareContext(&I->Obj, info);
  ramp = ColorGetRamp(G, I->Obj.Color);
  color = ColorGet(G, I->Obj.Color);

  if(!I->State)
    return;

  if(pass || info->ray) {
    if((I->Obj.visRep & cRepCGOBit)) {
      for(StateIterator iter(G, I->Obj.Setting, state, I->NState); iter.next();) {
        sobj = I->State + iter.state;
        if (!sobj->origCGO)
          continue;
        if (!ray)
          ObjectCGOGenerateCGO(G, I, sobj, use_shader, cgo_lighting, color, ramp, iter.state);	
	ObjectCGORenderState(G, pass, ray, pick, I, info, sobj, color, ramp, use_shader, cgo_lighting);
      }
    }
  }
}


/*========================================================================*/
ObjectCGO *ObjectCGONew(PyMOLGlobals * G)
{
  OOAlloc(G, ObjectCGO);

  ObjectInit(G, (CObject *) I);

  I->State = VLACalloc(ObjectCGOState, 10);
  I->NState = 0;
  I->Obj.type = cObjectCGO;
  I->Obj.fFree = (void (*)(CObject *)) ObjectCGOFree;
  I->Obj.fUpdate = (void (*)(CObject *)) ObjectCGOUpdate;
  I->Obj.fInvalidate = (void (*)(CObject *, int rep, int level, int state))
    ObjectCGOInvalidate;
  I->Obj.fRender = (void (*)(CObject *, RenderInfo *)) ObjectCGORender;
  I->Obj.fGetNFrame = (int (*)(CObject *)) ObjectCGOGetNState;

  return (I);
}


/*========================================================================*/
static CGO *ObjectCGOPyListFloatToCGO(PyMOLGlobals * G, PyObject * list)
{
  CGO *cgo = NULL;
  int len;
  int ok = true;
  int result;
  float *raw = NULL;
  if(PyList_Check(list)) {
    len = PConvPyListToFloatArray(list, &raw);
    if(len < 0)
      len = 0;
    if(raw) {
      if(ok) {
        cgo = CGONewSized(G, len);
        if(cgo) {
          result = CGOFromFloatArray(cgo, raw, len);
          if(result) {
            PRINTF " FloatToCGO: error encountered on element %d\n", result ENDF(G);
          }
          CGOStop(cgo);
        }
      }
      FreeP(raw);
    }
  }
  return (cgo);
}


/*========================================================================*/
static CGO *ObjectCGOFloatArrayToCGO(PyMOLGlobals * G, float *raw, int len, int quiet)
{
  CGO *cgo = NULL;
  int ok = true;
  int result;

  if(raw) {
    if(ok) {
      cgo = CGONewSized(G, len);
      if(cgo) {
        result = CGOFromFloatArray(cgo, raw, len);
        if(result && !quiet) {
          PRINTF " FloatToCGO: error encountered on element %d\n", result ENDF(G);
        }
        CGOStop(cgo);
      }
    }
  }
  return (cgo);
}


/*========================================================================*/
ObjectCGO *ObjectCGOFromCGO(PyMOLGlobals * G, ObjectCGO * obj, CGO * cgo, int state)
{
  ObjectCGO *I = NULL;

  if(obj) {
    if(obj->Obj.type != cObjectCGO)     /* TODO: handle this */
      obj = NULL;
  }
  if(!obj) {
    I = ObjectCGONew(G);
  } else {
    I = obj;
  }
  if(state < 0)
    state = I->NState;
  if(I->NState <= state) {
    VLACheck(I->State, ObjectCGOState, state);
    I->NState = state + 1;
  }

  CGOFree(I->State[state].renderCGO);
  CGOFree(I->State[state].origCGO);
  I->State[state].origCGO = cgo;

  if(I) {
    ObjectCGORecomputeExtent(I);
  }
  SceneChanged(G);
  SceneCountFrames(G);
  return (I);
}


/*========================================================================*/

ObjectCGO *ObjectCGONewVFontTest(PyMOLGlobals * G, const char *text, float *pos)
{

  ObjectCGO *I = NULL;
  int font_id;
  CGO *cgo = NULL;
  float scale[2] = { 1.0, 1.0 };

  font_id = VFontLoad(G, 1, 1, 1, true);
  cgo = CGONew(G);
  VFontWriteToCGO(G, font_id, cgo, text, pos, scale, NULL, NULL);
  I = ObjectCGOFromCGO(G, NULL, cgo, 0);

  return (I);
}


/*========================================================================*/
ObjectCGO *ObjectCGODefine(PyMOLGlobals * G, ObjectCGO * obj, PyObject * pycgo, int state)
{                               /* assumes blocked interpreter */
  ObjectCGO *I = NULL;

  CGO *cgo, *font_cgo;
  int est;

  if(obj) {
    if(obj->Obj.type != cObjectCGO)     /* TODO: handle this */
      obj = NULL;
  }
  if(!obj) {
    I = ObjectCGONew(G);
  } else {
    I = obj;
  }
  if(state < 0)
    state = I->NState;
  if(I->NState <= state) {
    VLACheck(I->State, ObjectCGOState, state);
    I->NState = state + 1;
  }

  CGOFree(I->State[state].origCGO);

  if(PyList_Check(pycgo)) {
    if(PyList_Size(pycgo)) {
      if(PyFloat_Check(PyList_GetItem(pycgo, 0))) {
        cgo = ObjectCGOPyListFloatToCGO(G, pycgo);
        if(cgo) {
          est = CGOCheckForText(cgo);
          if(est) {
            CGOPreloadFonts(cgo);
            font_cgo = CGODrawText(cgo, est, NULL);
            CGOFree(cgo);
            cgo = font_cgo;
          }
          est = CGOCheckComplex(cgo);
	  I->State[state].origCGO = cgo;
        } else {
          ErrMessage(G, "ObjectCGO", "could not parse CGO List.");
        }
      }
    }
  }
  if(I) {
    ObjectCGORecomputeExtent(I);
  }
  SceneChanged(G);
  SceneCountFrames(G);
  return (I);
}

ObjectCGO *ObjectCGOFromFloatArray(PyMOLGlobals * G, ObjectCGO * obj,
                                   float *array, int size, int state, int quiet)
{
  ObjectCGO *I = NULL;

  CGO *cgo, *font_cgo;
  int est;

  if(obj) {
    if(obj->Obj.type != cObjectCGO)     /* TODO: handle this */
      obj = NULL;
  }
  if(!obj) {
    I = ObjectCGONew(G);
  } else {
    I = obj;
  }
  if(state < 0)
    state = I->NState;
  if(I->NState <= state) {
    VLACheck(I->State, ObjectCGOState, state);
    I->NState = state + 1;
  }

  CGOFree(I->State[state].renderCGO);
  CGOFree(I->State[state].origCGO);

  cgo = ObjectCGOFloatArrayToCGO(G, array, size, quiet);
  if(cgo) {
    est = CGOCheckForText(cgo);
    if(est) {
      CGOPreloadFonts(cgo);
      font_cgo = CGODrawText(cgo, est, NULL);
      CGOFree(cgo);
      cgo = font_cgo;
    }
    est = CGOCheckComplex(cgo);
    I->State[state].origCGO = cgo;
  } else if(!quiet) {
    ErrMessage(G, "ObjectCGO", "could not parse CGO.");
  }
  if(I) {
    ObjectCGORecomputeExtent(I);
  }
  SceneChanged(G);
  SceneCountFrames(G);
  return (I);
}


